#!/bin/bash
# Version 3.1
#
# "THE COFFEE-WARE LICENSE" (Revision 12/2007):
#
#   Sebastian Wicki <gandro@gmx.net> wrote this file. As long as you retain
#   this notice you can do whatever you want with this stuff. If we meet some
#   day, and you think this stuff is worth it, you can buy me a cup of coffee
#   in return.
#

################################################################################
# DEFAULT SETTINGS #############################################################
################################################################################

# Premium Account User (optional)
PREMIUM_USER=
# Premium Account Password (optional, will be prompted if not given)
PREMIUM_PASS=

# Where to log any messages
LOGFILE="/mnt/www/tmp/out.txt"

# Fetch and save the MD5 hash of the downloads (1=yes or 0=no)
MD5SUM=0

# Where to save temporary data
TMPFILE="/mnt/www/tmp/rstmp_$$"

# Where to save downloaded files
DOWN="/mnt/www/down/"

# After Download halt system (1=yes or 0=no)
FINISH=1

# Command which will be executed instead of waiting 15min as free-user (optional)
IP_RENEWAL_CMD=

# Arguments for wget for the file download (optional)
WGET_ARGS=

################################################################################
################################################################################
################################################################################

#
# RSFileInfo $URL
#
# Get file information from the Rapidshare API
#
# Arguments:
#   $URL  - Rapidshare URL of the file to check
#
# Returns:
#   - return value 0 on success, non-zero on failure.
#   - an array containing the following elements on stdout:
#
#     $download_url, $filename, $filesize, $md5sum
#
function RSFileInfo() {
    local fileid=$(basename $(dirname "${1}"))
    local filename=$(basename "${1}")

    local apiurl="http://api.rapidshare.com/cgi-bin/rsapi.cgi?sub=checkfiles_v1"
    apiurl="${apiurl}&files=${fileid}&filenames=${filename}&incmd5=${MD5SUM}"

    local return=$(wget -q -O - "${apiurl}")
    local -a return_r=( `echo $return | tr ',' ' '` )

    if [ "${return:0:5}" = "ERROR" ] ; then
        echo "Rapidshare API Error:" ${return:7}
        return 1
    elif [ "${return_r[4]}" -ne 1 ] ; then
        case "${return_r[4]}" in
            0)
                echo "Error: File not found!"
                return 1
                ;;
            3)
                echo "Error: Server down!"
                return 1
                ;;
            4)
                echo "Error: File marked as illegal!"
                return 1
                ;;
            5)
                echo "Error: Anonymous file locked, because it has more than 10 downloads already"
                return 1
                ;;
            2|6)
                echo "Notice: TrafficShare Download"
                ;;
        esac

    fi

    local filename=${return_r[1]}
    local fileid=${return_r[0]}
    local filesize=${return_r[2]}
    local md5sum=$(echo ${return_r[6]} | tr '[:upper:]' '[:lower:]')

    local download_url="http://rs${return_r[3]}${return_r[5]}.rapidshare.com/files/"
    download_url="${download_url}${fileid}/${filename}"

    echo "${download_url} ${filename} ${filesize} ${md5sum}"
    return 0
}

#
# RSDownloadPA $URL $File
#
# Rapidshare Download as Premium User
#
# Arguments:
#   $URL  - Rapidshare-Mirror URL of the file to check
#   $File - Download file destination
#
# Returns:
#   - return value 0 on success, non-zero on failure
#
function RSDownloadPA() {
    local url=${1}
    local file=${2}
    local wgetarg=$WGET_ARGS

    if [ "${DIRECTSTART:-0}" = 0 ] ; then
        # force directstart
        url="${url}?directstart=1"
    fi

    if [ -e "${file}" ] ; then
        wgetarg="${wgetarg} --continue"
    fi

    wget --header="Cookie: enc=${COOKIE}" ${wgetarg} -O "${DOWN}${file}" "${url}"
    return $?
}

#
# RSParseHTMLFree $URL
#
# Parse HTML output for free users
#
# Arguments:
#   $URL  - Rapidshare-Mirror URL of the file
#
# Returns:
#   - return value 0 on success, non-zero on failure
#   - Free user download URL on stdout
#
function RSParseHTMLFree() {
    local download_url
    local wait

    while true ; do
        wget -q --post-data="dl.start=Free" -O "${TMPFILE}" "${1}"

        wait=$(sed -n -e 's/^var c=\([0-9]*\);.*/\1/p' "${TMPFILE}")
        download_url=$(grep 'name="dlf"' "${TMPFILE}" | \
                         sed -n -e 's/.*action="\(.*\)" method.*/\1/p')

        if [ -n "${wait}" -a -n "${download_url}" ] ; then
            echo "Ready. Waiting ${wait} seconds..." >&2
            sleep "${wait}"
            break
        elif grep -q 'download limit' "${TMPFILE}" ; then
            if [ -z "${IP_RENEWAL_CMD}" ] ; then
                wait=$(sed -n -e 's/.*in about \([0-9]*\) minutes.*/\1/p' "${TMPFILE}")
                echo "Download limit reached. Waiting ${wait} minutes..." >&2
                sleep "${wait}m"
            else
                ${IP_RENEWAL_CMD}
            fi
        elif grep -q -e 'a lot of users' -e 'no more download slots' \
             -e 'servers are overloaded' "${TMPFILE}"
        then
            echo "No more download slots. Waiting 2 minutes..." >&2
            sleep 2m
            continue
        elif grep -q 'is already downloading' "${TMPFILE}" ; then
            echo "IP Address is already downloading. Waiting 30 minutes..." >&2
            sleep 30m
            continue
        else
            local dumpfile="rserror_${filename}.html"
            cp "${TMPFILE}" "${dumpfile}"
            echo "Error: Unknown Rapidshare error. \""${dumpfile}"\" dumped.">&2
            return 1
        fi

    done

    echo "${download_url}"
    return 0
}

#
# RSParseHTMLFree $URL $File
#
# Rapidshare Download as Free User
#
# Arguments:
#   $URL  - Rapidshare-Mirror URL of the file
#   $File - Download file destination
#
# Returns:
#   - return value 0 on success, non-zero on failure
#
function RSDownloadFree() {
    local url=$1
    local filename=$2
    local download_url

    if ! download_url=$(RSParseHTMLFree ${url}) ; then
        return 1
    elif [ -e "${filename}" ] ; then
        echo -n "Warning: ${filename} already exists - " >&2
        filename=$(mktemp -u "${filename}.XXXX")
        echo "saving as ${filename}" >&2
    fi

    wget ${WGET_ARGS} -a "${LOGFILE}" -O "${DOWN}${filename}" "${download_url}"
    return $?
}

#
# RSLoginPA $User $Password
#
# Rapidshare Premium User Login
#
# Arguments:
#   $User     - Premium User
#   $Password - Premium Password
#
# Returns:
#   - return value 0 on success, non-zero on failure
#
function RSLoginPA() {
    local apiurl="https://api.rapidshare.com/cgi-bin/rsapi.cgi?"
    apiurl="${apiurl}sub=getaccountdetails_v1&withcookie=1&type=prem&login=${1}&password=${2}"
    local return=$(wget -q -O - "$apiurl")

    if [ "${return:0:5}" = "ERROR" ] ; then
        echo "Error:" ${return:7}
        return 1
    fi

    COOKIE=$(echo $return | sed -n 's/.*cookie=\([[:xdigit:]]\+\).*/\1/p')
    DIRECTSTART=$(echo $return | sed -n 's/.*directstart=\([0-1]\).*/\1/p')

    if [ -z "${COOKIE}" ] ; then
        echo "Error: Cookie not found!"
        return 1
    fi

    return 0
}

#
# RSDownloadFile $URL $File
#
# Start Rapidshare Download (as free- or premium-user)
#
# Arguments:
#   $URL  - Rapidshare URL of the file
#
# Returns:
#   - return value 0 on success, non-zero on failure
#
function RSDownloadFile() {
    local -a fileinfo

    fileinfo=( `RSFileInfo $1` )
    if [ $? -gt 0 ] ; then
        # print error msg
        echo ${fileinfo[@]}
        return 1
    fi

    local filesize=$(expr ${fileinfo[2]} / 1024 / 1024)
    echo "Downloading \"${fileinfo[1]}\" (${filesize} MiB)" >&2

    if [ "${MD5SUM}" = 1 ] ; then
        echo "${fileinfo[3]}  ${fileinfo[1]}" >>MD5SUMS
    fi

    if [ -n "${PREMIUM_USER}" ] && [ -n "${PREMIUM_PASS}" ] ; then
        RSDownloadPA ${fileinfo[0]} ${fileinfo[1]}
    else
        RSDownloadFree ${fileinfo[0]} ${fileinfo[1]}
    fi
    return $?
}


#
# Main programm
#

while [ -n "${1}" ] ; do
    case ${1} in
        -l|--log)
            LOGFILE=${2?"Missing logfile"}
            shift
            ;;
        -n|--no-md5)
            MD5SUM=0
            ;;
        -i|--input)
            INPUT=${2?"Missing inputfile"}
            shift
            ;;
        -u|--user)
            PREMIUM_USER=${2?"Missing username"}
            shift
            ;;
        -p|--pass)
            PREMIUM_PASS=${2?"Missing password"}
            shift
            ;;
        http://*)
            URL=${1}
            ;;
        -h|--help)
            echo "Usage:"
            echo
            echo "  ${0} [OPTIONS] [URL]" >&2
            echo
            echo "Available Options:"
            echo "    -i --input  INPUTFILE   Read urls from input file"
            echo "    -l --log    LOGFILE     Write output to log file"
            echo "    -n --no-md5             Don't create MD5SUMS file"
            echo
            echo "    -u --user USERNAME      Premium account username"
            echo "    -p --pass PASSWORD      Premium account password"
            echo
            echo "    -h --help               Show this help page"
            echo
            exit
            ;;
        *)
            echo "${0}: unknown option: ${1}" >&2
            ;;
    esac
    shift
done

# Initialize logfile
exec 2>>${LOGFILE}

# Clear out.txt that gives a status via web interface
if [ -f /mnt/www/links/out.txt ] ; then
    rm /mnt/www/links/out.txt
fi

if [ -f "${LOGFILE}" ] ; then
    WGET_ARGS="--progress=bar:force ${WGET_ARGS}"
else
    WGET_ARGS="--progress=bar:force ${WGET_ARGS}"
fi

# Delete temporary data afterwards
trap "rm ${TMPFILE} 2>/dev/null; exit" EXIT SIGINT SIGTERM SIGKILL

# Premium Account login, if necessary
if [ -n "${PREMIUM_USER}" ] ; then
    echo "Notice: Logging in as premium user \"${PREMIUM_USER}\"... "
    if [ -z "${PREMIUM_PASS}" ] ; then
        read -s -p "RSPA Password: " PREMIUM_PASS
    fi

    if ! RSLoginPA ${PREMIUM_USER} ${PREMIUM_PASS} ; then
        exit 1
    fi
fi


if [ -z "${URL}" ] && [ -z "${INPUT}" ] ; then
    echo "Error: No url or inputfile specified." >&2
    exit 1
fi

# Start downloading one URL
if [ -n "${URL}" ] ; then
    RSDownloadFile "${URL}"
fi

# Or start reading from the input list
if [ -n "${INPUT}" ] ; then
    while read url ; do
        if [ "${url:0:4}" = 'http' ] ; then
            RSDownloadFile "${url}"
        fi
    done < "${INPUT}"
fi

# Close logfile
exec 2>&-
# Remove out.txt set by web interface
if [ -f /mnt/www/tmp/out.txt ] ; then
    rm /mnt/www/tmp/out.txt
fi

# Switch off PC 
if [ "${FINISH}" = 1 ] ; then
        poweroff -f
fi
