/************************************************************************

    WIN1VGA v1.0.0 - Patch the Windows 1.x EGA driver and setup program
                     for VGA 640x480 and VESA 800x600 modes.

    Copyright (C) 2011  John Elliott <jce@seasip.demon.co.uk>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

*************************************************************************/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>

/* Assume all the files we are patching fit in a 40000-byte buffer. */
#define BUFFSIZE 40000
typedef unsigned char byte;

byte buf[BUFFSIZE];
unsigned drvsize;

#define EGAMODE  0x10	/* EGA  640x350 */
#define VGAMODE  0x12	/* VGA  640x480 */
#define VESAMODE 0x6A	/* VESA 800x600 */

/* The patches. Each patch comes in two or three sets: the byte sequence
 * to search for, and one or two sequences to replace it with. */

/* First patch: Setting the video mode. Replace */

byte find1[] = { 0xCD, 0x10, 		/* INT 10h */
                 0xAA, 			/* STOSB */
                 0xB8, EGAMODE, 0x00, 	/* MOV AX, EGAMODE */
                 0xCD, 0x10, 		/* INT 10h */
                 0xB8, 0x00, 0x0F, 	/* MOV AX, 0F00h */
                 0xCD, 0x10, 		/* INT 10h */
                 0x3C, EGAMODE };	/* CMP AL, EGAMODE */

byte vga1[]  = { 0xCD, 0x10, 		/* As above, but for VGAMODE */
                 0xAA, 
                 0xB8, VGAMODE, 0x00, 
                 0xCD, 0x10, 
                 0xB8, 0x00, 0x0F, 
                 0xCD, 0x10, 
                 0x3C, VGAMODE };

byte vesa1[]  = { 0xCD, 0x10, 		/* As above, but for VESAMODE */
                 0xAA, 
                 0xB8, VESAMODE, 0x00, 
                 0xCD, 0x10, 
                 0xB8, 0x00, 0x0F, 
                 0xCD, 0x10, 
                 0x3C, VESAMODE };

/* Second patch: Device properties table. First word is video segment and
 * remains unchanged; second, third and fourth are width, height, bytes per
 * row */
byte find2a[] = { 0x00, 0xA0, 0x80, 0x02, 0x5E, 0x01, 0x50, 0x00 };/* 640x350 */
byte vga2a [] = { 0x00, 0xA0, 0x80, 0x02, 0xE0, 0x01, 0x50, 0x00 };/* 640x480 */
byte vesa2a[] = { 0x00, 0xA0, 0x20, 0x03, 0x58, 0x02, 0x64, 0x00 };/* 800x600 */

/* Third patch: GDIINFO structure. First two words are aspect ratio,
 * second two are width and height */
byte find2b[] = { 0xF0, 0x00, 0xAF, 0x00, 0x80, 0x02, 0x5E, 0x01 };/* 640x350 */
byte vga2b [] = { 0xF0, 0x00, 0xF0, 0x00, 0x80, 0x02, 0xE0, 0x01 };/* 640x480 */
byte vesa2b[] = { 0xF0, 0x00, 0xF0, 0x00, 0x20, 0x03, 0x58, 0x02 };/* 800x600 */

/* Fourth patch: Vertical resolution is hardcoded at one point  */
byte find2c[] = { 0x81, 0xE9, 0x5E, 0x01 };	/* SUB CX, 350 */
byte vga2c [] = { 0x81, 0xE9, 0xE0, 0x01 };	/* SUB CX, 640 */
byte vesa2c[] = { 0x81, 0xE9, 0x58, 0x02 };	/* SUB CX, 600 */

/* The next series of patches are only for the VESA driver, and deal with
 * drawing the mouse pointer; the code to do this hardcodes the number of
 * bytes per row in a number of places */
byte find2d[] = { 0xb8, 0x47, 0x00 };	/* MOV AX, 80-9 */
byte vesa2d[] = { 0xb8, 0x5B, 0x00 };	/* MOV AX, 100-9 */
byte find2e[] = { 0xb8, 0x4B, 0x00 };	/* MOV AX, 80-5 */
byte vesa2e[] = { 0xb8, 0x5F, 0x00 };	/* MOV AX, 100-5 */
byte find2f[] = { 0x83, 0xE9, 0x50 };	/* SUB CX, 80 */
byte vesa2f[] = { 0x83, 0xE9, 0x64 };	/* SUB CX, 100 */

/* Patch the subroutine that calculates the position of the pointer in 
 * video RAM. Originally this would use shifts and multiplies to add AX*50h 
 * to SI. Calculating AX*64h the same way would require more space, so bite 
 * the bullet and use MUL. */
byte find2h[] = { 0x03, 0xC0, 	/* ADD AX, AX */
		  0x03, 0xC0, 	/* ADD AX, AX */
		  0x03, 0xC0, 	/* ADD AX, AX */
		  0x03, 0xC0,	/* ADD AX, AX */
		  0x03, 0xF0 	/* ADD SI, AX */
};
byte vesa2h[] = { 0x52, 		/* PUSH DX */
		  0xba, 0x64, 0x00, 	/* MOV DX, 100 */
		  0xF7, 0xE2, 		/* MUL DX */
		  0x5A, 		/* POP DX */
		  0x03, 0xF0, 		/* ADD SI, AX */
		  0xC3 			/* RET */
};


/* The driver hardcodes the size of a plane of video RAM here and there */
byte find3[] = { 0x60, 0x6D };	/* plane size 28000 */ 
byte vga3[]  = { 0x00, 0x96 };	/* plane size 38400 */
byte vesa3[] = { 0x60, 0xEA };	/* plane size 60000 */

byte find4[] = { 0x00, 0x6E };	/* plane size+2 rows 28160 */
byte vga4[]  = { 0xA0, 0x96 };	/* plane size+2 rows 38560 */
byte vesa4[] = { 0x28, 0xEB };	/* plane size+2 rows 60200 */

/* These patches are applied to the setup program, to replace EGAHIRES.DRV 
 * and EGAHIBW.DRV with VGAHIRES.DRV and VESA800.DRV */
byte find5[] = "EGAHIRES.DRV";
byte vga5[]  = "VGAHIRES.DRV";

byte find6[] = "EGA (more than 64K)";
byte vga6[]  = "VGA 640x480\0_______";

byte find8[] = "EGAHIBW.DRV";
byte vesa8[] = "VESA800.DRV";

byte find9[] = "EGA with Enhanced Color Display (";
byte vesa9[] = "VESA 800x600\0____________________";

/* Apply a patch to the buffer */
void replace(const char *txt, byte *a, byte *b, int len)
{
    unsigned n;
    for (n = 0; n <= drvsize-len; n++)
    {
	/* Search for 'find' pattern */
        if (!memcmp(buf+n, a, len))
        {
	    /* If it's found, log it and substitute 'replace' pattern */
            printf("Patching %s at %04x\n", txt, n);
            memcpy(buf+n, b, len);
        }
    } 
}

/* Copy a file. Used to create .GRB and .LGO files to go with the driver. */
int copy(const char *s, const char *d)
{
    FILE *fp1, *fp2;
    int c;

    fp1 = fopen(s, "rb");
    if (!fp1)
    {
        fprintf(stderr, "Cannot open %s\n", s);
        return 1; 
    }
    fp2 = fopen(d, "wb");
    if (!fp2)
    {
        fprintf(stderr, "Cannot open %s\n", d);
        fclose(fp1);
        return 1; 
    }
    while ( (c = fgetc(fp1)) != EOF)
    {
        fputc(c, fp2);
    }
    fclose(fp2);
    fclose(fp1);
    return 0;
} 

/* Create VGAHIRES.DRV */
void pat_vga(void)
{
    printf("Creating VGAHIRES.DRV from EGAHIRES.DRV:\n");
    replace("mode set",     find1,  vga1,  sizeof(find1));
    replace("resolution",   find2a, vga2a, sizeof(find2a));
    replace("resolution",   find2b, vga2b, sizeof(find2b));
    replace("resolution",   find2c, vga2c, sizeof(find2c));
    replace("plane size",   find3,  vga3,  sizeof(find3));
    replace("plane size",   find4,  vga4,  sizeof(find4));
}


/* Create VESA800.DRV */
void pat_vesa(void)
{
    printf("Creating VESA800.DRV from EGAHIRES.DRV:\n");
    replace("mode set",     find1,  vesa1,  sizeof(find1));
    replace("resolution",   find2a, vesa2a, sizeof(find2a));
    replace("resolution",   find2b, vesa2b, sizeof(find2b));
    replace("resolution",   find2c, vesa2c, sizeof(find2c));
    replace("x resolution", find2d, vesa2d, sizeof(find2d));
    replace("x resolution", find2e, vesa2e, sizeof(find2e));
    replace("x resolution", find2f, vesa2f, sizeof(find2f));
    replace("concat",       find2h, vesa2h, sizeof(find2h));
    replace("plane size",   find3,  vesa3,  sizeof(find3));
    replace("plane size",   find4,  vesa4,  sizeof(find4));
}


/* Patch the setup program */
void pat_setup(void)
{
    printf("Creating VGASETUP.EXE from SETUP.EXE:\n");
    replace("driver name", find5, vga5,  sizeof(find5)-1); 
    replace("driver desc", find6, vga6,  sizeof(find6)-1); 
    replace("driver name", find8, vesa8, sizeof(vesa8)-1); 
    replace("driver desc", find9, vesa9, sizeof(vesa9)-1); 
}

/* Load EGAHIRES.DRV, patch it, write out VGAHIRES.DRV */
int fixvga()
{
    FILE *fp;

    memset(buf, 0, sizeof(buf));
    fp = fopen("egahires.drv", "rb");
    if (fp == NULL)
    {
        fprintf(stderr, "Cannot open EGAHIRES.DRV for read.\n");
        return 1;
    }
    drvsize = fread(buf, 1, sizeof(buf), fp); 
    fclose(fp);

    pat_vga();

    fp = fopen("vgahires.drv", "wb");
    if (fp == NULL)
    {
        fprintf(stderr, "Cannot open VGAHIRES.DRV for write.\n");
        return 1;
    }
    if (fwrite(buf, 1, drvsize, fp) != drvsize)
    {
        fprintf(stderr, "Could not write to VGAHIRES.DRV.\n");
        fclose(fp);
        return 1;
    }
    fclose(fp);
    return 0;
}

/* Load EGAHIRES.DRV, patch it, write out VESA800.DRV */
int fixvesa()
{
    FILE *fp;

    memset(buf, 0, sizeof(buf));
    fp = fopen("egahires.drv", "rb");
    if (fp == NULL)
    {
        fprintf(stderr, "Cannot open EGAHIRES.DRV for read.\n");
        return 1;
    }
    drvsize = fread(buf, 1, sizeof(buf), fp); 
    fclose(fp);

    pat_vesa();

    fp = fopen("vesa800.drv", "wb");
    if (fp == NULL)
    {
        fprintf(stderr, "Cannot open VESA800.DRV for write.\n");
        return 1;
    }
    if (fwrite(buf, 1, drvsize, fp) != drvsize)
    {
        fprintf(stderr, "Could not write to VGAHIRES.DRV.\n");
        fclose(fp);
        return 1;
    }
    fclose(fp);
    return 0;
}


/* Load SETUP.EXE, patch it, write out VGASETUP.EXE */
int fixsetup()
{
    FILE *fp;

    memset(buf, 0, sizeof(buf));
    fp = fopen("setup.exe", "rb");
    if (fp == NULL)
    {
        fprintf(stderr, "Cannot open SETUP.EXE for read.\n");
        return 1;
    }
    drvsize = fread(buf, 1, sizeof(buf), fp); 
    fclose(fp);

    pat_setup();

    fp = fopen("vgasetup.exe", "wb");
    if (fp == NULL)
    {
        fprintf(stderr, "Cannot open VGASETUP.EXE for write.\n");
        return 1;
    }
    if (fwrite(buf, 1, drvsize, fp) != drvsize)
    {
        fprintf(stderr, "Could not write to VGASETUP.EXE.\n");
        fclose(fp);
        return 1;
    }
    fclose(fp);
    return 0;
}


int main(int argc, char **argv)
{
    if (fixvga()) return 1;
    if (fixvesa()) return 1;
    if (fixsetup()) return 1;
    if (copy("egahires.grb", "vgahires.grb")) return 1;
    if (copy("egahires.lgo", "vgahires.lgo")) return 1;
    if (copy("egahires.grb", "vesa800.grb")) return 1;
    if (copy("egahires.lgo", "vesa800.lgo")) return 1;
    return 0;
}
