import re
import sys
import json
import string
import logging

from wright import WrightSystem

vacancies = 6
valid_candidates = ['CHRiSNEW', 'Chrisse', 'CoburgM', 'CommodoreFreak',
                    'Der Doktor', 'DosAmp', 'HOMEBOY87', 'Spit', 'Wynton',
                    'Xaar', 'meego4ever!', 'michi', 'mrshadowtux', 'oreissig',
                    'thosch97', 'winfreak', 'xCtrl']


rank_regexp = re.compile("\[(?P<pref>\d+)\]")
vote_regexp = re.compile("\[(?P<pref>\d+)\] (?P<cand>[A-Za-z0-9 !-]+)$")

def validate_vote(rawvote):
    ranks = rank_regexp.findall(rawvote)
    if not ranks:
        logging.warning("Missing ranks: {}".format(rawvote.replace('\n', ' ')))
        return False

    prev_rank = 0
    for rank in sorted(map(int, ranks)):
        if prev_rank+1 != rank:
            logging.warning("Non-continuous rank {}: {}".
                format(rank, rawvote.replace('\n', ' ')))
            return False

        prev_rank = rank

    return True

def parse_vote(votelist):
    preference = {}
    for vote in votelist.splitlines():
        m = vote_regexp.match(vote)
        if m:
            pref = int(m.group("pref")) - 1
            cand = m.group("cand")
            preference[pref] = cand

    return [preference[i] for i in sorted(preference)]
#
# main
#

logging.basicConfig(filename='whf.log', filemode='w', level=logging.INFO)

if len(sys.argv) < 2:
    sys.exit("{}: INPUT".format(sys.argv[0]))

# read results from file
with open(sys.argv[1]) as i:
    data = json.load(i)

# strip away metadata
results = data['root']

# remove invalid characters from votes
allowed = string.ascii_letters + string.digits + string.whitespace + "![]"
raw_votes = ["".join(c for c in r['vote'] if c in allowed) for r in results]

valid_raw_votes = filter(validate_vote, raw_votes)

candiates = set()
ballots = {}
for i, vote in enumerate(valid_raw_votes):
    preference = parse_vote(vote)
    if not all(c in valid_candidates for c in preference):
        logging.warning("Invalid candidates: {}".format(vote.replace('\n', '')))
        continue
    ballots[i] = preference



ws = WrightSystem(vacancies, ballots)
winners = ws.get_winners()
report = ws.get_report()

with open("report.json", "w") as r:
    json.dump(report, r, sort_keys=True, indent=4)

for winner in sorted(winners):
    print(winner)
